// Config
const MAPBOX_ACCESS_TOKEN = "";
const GEOSPAN_API_TOKEN = "";

// Constants
const MARKETPLACE_API_HOME = "https://api.geospan.com/remote4d/v1/api";
const FOOTPRINTS_URL = `${MARKETPLACE_API_HOME}/spatial/footprints`;
const ESTIMATE_URL = `${MARKETPLACE_API_HOME}/gsquare/estimate`;
const ESTIMATE_RESULT_URL = `${MARKETPLACE_API_HOME}/gsquare/query`;
const AUTH_HEADER = { Authorization: `Api-Key ${GEOSPAN_API_TOKEN}` };
const RESULTS_MAX_ATTEMPTS = 20;
const RESULTS_CHECK_DELAY = 2000;
const SQ_FT_PER_SQ_M = 10.7639;

mapboxgl.accessToken = MAPBOX_ACCESS_TOKEN;

// Initialize our map
const map = new mapboxgl.Map({
  container: "map", // container ID
  center: [-74.568, 39.983], // starting position [lng, lat]
  zoom: 19, // starting zoom
  style: "mapbox://styles/mapbox/satellite-v9",
});

// Once the map loads, set up our datasources and layers
map.on("load", () => {
  // Create a new data source with an empty feature collection
  map.addSource("building-footprints", {
    type: "geojson",
    data: {
      type: "FeatureCollection",
      features: [],
    },
    // Make sure all features have a unique ID
    generateId: true,
  });

  // Add a new layer that uses our new data source to create features visible in
  // the map
  map.addLayer({
    id: "building-footprints-layer",
    type: "fill",
    source: "building-footprints",
    layout: {},
    paint: {
      // Conditional fill-color formatting for highlighting when clicked
      "fill-color": [
        "case",
        ["boolean", ["feature-state", "click"], false],
        "#FFFF00",
        "#0080ff",
      ],
      "fill-opacity": 0.5,
    },
  });

  // Load our footprint data for the first time
  loadNewFootprints();
});

// takes a polygon feature and returns its center coordinate as a WKT string
const getCenterWKT = (feature) => {
  const bounds = new mapboxgl.LngLatBounds();
  feature.geometry.coordinates[0].forEach((coordinate) => {
    bounds.extend(coordinate);
  });
  const center = bounds.getCenter();
  return `POINT (${center.lng} ${center.lat})`;
};

// Given the current bounds of the map, load building footprints for that area
// and add them into our building-footprints datasource
const loadNewFootprints = async () => {
  const bounds = map.getBounds().toArray().flat();
  const params = new URLSearchParams({
    bounds,
  }).toString();

  const footprints = await fetch(`${FOOTPRINTS_URL}?${params}`, {
    headers: AUTH_HEADER,
  })
    .then((data) => data.json())
    .catch((e) => displayResults("Could not load footprints"));

  if (footprints) {
    map.getSource("building-footprints").setData(footprints);
  }
};

// Submits and retrieves the results of the estimate
const requestEstimate = async (wkt) => {
  // Submit estimate request
  const { queryKey } = await fetch(ESTIMATE_URL, {
    method: "POST",
    headers: {
      ...AUTH_HEADER,
      Accept: "application/json",
      "Content-Type": "application/json",
    },
    body: JSON.stringify({
      wkt,
      includeImagery: true,
      includeWeather: true,
    }),
  }).then((data) => data.json());

  // Poll Geospan for results
  let counter = RESULTS_MAX_ATTEMPTS;
  return new Promise((resolve, reject) => {
    const loadResults = () => {
      fetch(`${ESTIMATE_RESULT_URL}/${queryKey}`, {
        headers: AUTH_HEADER,
      })
        .then((data) => data.json())
        .then(({ state, results }) => {
          if (state === "SUCCESS") {
            resolve(results);
          } else if (state === "PENDING" && counter > 0) {
            setTimeout(() => {
              loadResults(queryKey);
            }, RESULTS_CHECK_DELAY);
          } else if (state === "FAILURE") {
            reject("There was an error in processing this request.");
          } else {
            reject("Your request has timed out.");
          }
          counter--;
        })
        .catch((e) => reject(e));
    };

    if (queryKey) {
      loadResults(queryKey);
    } else {
      reject(
        "There was an error creating the request. Please select another location or try again in a moment."
      );
    }
  });
};

// Displays the results in the sidebar
const displayResults = (results) => {
  document.querySelector("#results").innerHTML = results;
};

let highlightedBuildingId = null;

// Remove highlight from a building and clear results in the sidebar
const clearResults = () => {
  if (highlightedBuildingId !== null) {
    map.setFeatureState(
      { source: "building-footprints", id: highlightedBuildingId },
      { click: false }
    );
  }
  highlightedBuildingId = null;
  displayResults("");
};

// Adds a highlight to a clicked building
const highlightClickedBuilding = (mapboxFeature) => {
  clearResults();

  map.setFeatureState(
    { source: "building-footprints", id: mapboxFeature.id },
    { click: true }
  );

  highlightedBuildingId = mapboxFeature.id;
};

// Make sure that every time we pan / zoom the map, we update our building
// footprints
map.on("moveend", () => {
  clearResults();
  loadNewFootprints();
});

// When we click on a building, highlight the building and request an estimate
map.on("click", "building-footprints-layer", (e) => {
  highlightClickedBuilding(e.features[0]);

  displayResults("Loading gSquare Estimate...");

  // Get the center coordinate of the building and convert it to a WKT string
  // that we can use to submit our estimate request
  const wkt = getCenterWKT(e.features[0]);
  requestEstimate(wkt)
    .then((data) =>
      displayResults(`
            <div><b>Total Area</b>: ${(
              data.totalArea.area * SQ_FT_PER_SQ_M
            ).toFixed(2)} sq ft</div>
            <div><b>Primary Pitch</b>: ${data.pitchResult.primaryPitch}</div>
        `)
    )
    .catch((e) => displayResults("Unable to load estimate"));
});
